<?php
/**
 * Plugin Name: Advanced Registration Firewall
 * Version: 11.0
 * Description: Registration-only firewall with auto-ban, manual bans, whitelist, country allowlist, alerts, rotating logs, and forbidden keyword check. Whitelisted IPs do not accrue failure counts.
 */

if (!defined('ABSPATH')) exit;

/* ==================================================
 * ADMIN MENU
 * ================================================== */
add_action('admin_menu', function () {
    add_menu_page(
        'Advanced Registration Firewall',
        'Registration Firewall',
        'manage_options',
        'arf_admin',
        'arf_admin_page',
        'dashicons-shield-alt',
        30
    );
});

/* ==================================================
 * ADMIN PAGE
 * ================================================== */
function arf_admin_page() {
    global $wpdb;

    $logs_table    = $wpdb->prefix . 'arf_logs';
    $blocked_table = $wpdb->prefix . 'arf_blocked_ips';

    $per_page = 15;
    $current_page = max(1, intval($_GET['paged'] ?? 1));
    $offset = ($current_page - 1) * $per_page;

    $total_logs  = (int) $wpdb->get_var("SELECT COUNT(*) FROM $logs_table");
    $total_pages = max(1, ceil($total_logs / $per_page));
    ?>
    <div class="wrap">
        <h1>Advanced Registration Firewall</h1>

        <form method="post" action="options.php">
            <?php settings_fields('arf_options_group'); ?>

            <h2>Auto-Ban Settings</h2>
            <p>
                Automatically blocks an IP from <strong>registering only</strong> after repeated failed attempts.
                This never blocks site access.
            </p>
            <p><em>Tip:</em> 35 attempts is recommended.</p>
            <input type="number" min="1" name="arf_auto_ban_threshold"
                   value="<?php echo esc_attr(get_option('arf_auto_ban_threshold', 5)); ?>" />

            <h2>Admin Alert Emails</h2>
            <p>
                Email notifications sent when an IP is auto-banned.
                Each address is validated and notified once per ban.
            </p>
            <p><em>Tip:</em> Separate multiple emails with commas.</p>
            <input type="text" name="arf_admin_alert_emails"
                   value="<?php echo esc_attr(get_option('arf_admin_alert_emails')); ?>"
                   style="width:100%;" />

            <h2>Allowed Countries</h2>
            <p>
                Only users from these countries are allowed to register.
                Country codes must be ISO-2 format (e.g. US, CA, GB).
            </p>
            <p><em>Tip:</em> One country code per line. Leave empty to allow only US.</p>
            <textarea name="arf_allowed_countries" rows="4" style="width:100%;"><?php
                echo esc_textarea(get_option('arf_allowed_countries'));
            ?></textarea>

            <h2>IP Whitelist</h2>
            <p>Whitelisted IPs bypass manual IP block and auto-ban <strong>and do not accrue failure counts</strong>, but still must follow country, .com email, and keyword rules.</p>
            <textarea name="arf_ip_whitelist" rows="4" style="width:100%;"><?php
                echo esc_textarea(get_option('arf_ip_whitelist'));
            ?></textarea>

            <h2>Manual Email Ban</h2>
            <p>Immediately blocks specific email addresses from registering.</p>
            <textarea name="arf_email_banlist" rows="4" style="width:100%;"><?php
                echo esc_textarea(get_option('arf_email_banlist'));
            ?></textarea>

            <h2>Manual IP Ban</h2>
            <p>Immediately blocks specific IP addresses from registering.</p>
            <textarea name="arf_manual_ip_banlist" rows="4" style="width:100%;"><?php
                echo esc_textarea(get_option('arf_manual_ip_banlist'));
            ?></textarea>

            <h2>Forbidden Keywords</h2>
            <p>Prevent certain words from being used in usernames or emails.</p>
            <p><em>Tip:</em> One keyword per line. Case-insensitive.</p>
            <textarea name="arf_banned_keywords" rows="4" style="width:100%;"><?php
                echo esc_textarea(get_option('arf_banned_keywords'));
            ?></textarea>

            <?php submit_button(); ?>
        </form>

        <h2>Blocked IP Addresses</h2>
        <table class="widefat striped">
            <tr><th>IP Address</th><th>Action</th></tr>
            <?php
            $blocked = $wpdb->get_results("SELECT * FROM $blocked_table");
            if ($blocked) {
                foreach ($blocked as $row) {
                    $url = wp_nonce_url(
                        admin_url('admin-post.php?action=arf_unblock_ip&ip=' . urlencode($row->ip)),
                        'arf_unblock_ip'
                    );
                    echo "<tr>
                        <td>{$row->ip}</td>
                        <td><a class='button' href='$url'>Unblock</a></td>
                    </tr>";
                }
            } else {
                echo "<tr><td colspan='2'>No blocked IPs.</td></tr>";
            }
            ?>
        </table>

        <h2>Recent Registration Logs</h2>
        <table class="widefat striped">
            <tr>
                <th>Date</th><th>IP</th><th>Country</th><th>Username</th><th>Email</th><th>Status</th>
            </tr>
            <?php
            $logs = $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT * FROM $logs_table ORDER BY timestamp DESC LIMIT %d OFFSET %d",
                    $per_page,
                    $offset
                )
            );

            foreach ($logs as $log) {
                echo "<tr>
                    <td>{$log->timestamp}</td>
                    <td>{$log->ip}</td>
                    <td>{$log->country}</td>
                    <td>{$log->username}</td>
                    <td>{$log->email}</td>
                    <td>{$log->status}</td>
                </tr>";
            }
            ?>
        </table>

        <div style="margin-top:12px;">
            <?php
            for ($i = 1; $i <= $total_pages; $i++) {
                if ($i === $current_page) {
                    echo "<strong style='margin-right:6px;'>$i</strong>";
                } else {
                    echo "<a class='button' style='margin-right:4px;' href='?page=arf_admin&paged=$i'>$i</a>";
                }
            }
            ?>
        </div>
    </div>
<?php }

/* ==================================================
 * SETTINGS
 * ================================================== */
add_action('admin_init', function () {
    register_setting('arf_options_group', 'arf_auto_ban_threshold');
    register_setting('arf_options_group', 'arf_admin_alert_emails');
    register_setting('arf_options_group', 'arf_allowed_countries');
    register_setting('arf_options_group', 'arf_ip_whitelist');
    register_setting('arf_options_group', 'arf_email_banlist');
    register_setting('arf_options_group', 'arf_manual_ip_banlist');
    register_setting('arf_options_group', 'arf_banned_keywords');
});

/* ==================================================
 * HELPERS
 * ================================================== */
function arf_list_contains($option, $value) {
    $list = array_filter(array_map('trim', explode("\n", strtolower(get_option($option)))));
    return in_array(strtolower($value), $list, true);
}

/* ==================================================
 * REGISTRATION VALIDATION
 * ================================================== */
add_filter('registration_errors', 'arf_validate_registration', 5, 3);
add_filter('woocommerce_registration_errors', 'arf_validate_registration', 5, 3);

function arf_validate_registration($errors, $username, $email) {
    $ip = $_SERVER['REMOTE_ADDR'];
    $country = arf_get_country_from_ip($ip);
    $ip_whitelisted = arf_list_contains('arf_ip_whitelist', $ip);

    // COUNTRY BLOCK
    $allowed_countries = array_filter(array_map('trim', explode("\n", get_option('arf_allowed_countries'))));
    if ($allowed_countries && !in_array($country, $allowed_countries, true)) {
        $errors->add('arf_country_blocked', 'Registrations from your country are not allowed.');
        arf_log($ip, $username, $email, $country, 'Blocked: Country');
        return $errors;
    }

    // EMAIL MANUAL BAN
    if (arf_list_contains('arf_email_banlist', $email)) {
        $errors->add('arf_email_banned', 'This email address is not allowed.');
        arf_log($ip, $username, $email, $country, 'Manually Banned Email');
        return $errors;
    }

    // IP MANUAL BAN OR AUTO-BLOCK (skip only for whitelisted IPs)
    if (!$ip_whitelisted && (arf_list_contains('arf_manual_ip_banlist', $ip) || arf_is_ip_blocked($ip))) {
        $errors->add('arf_ip_banned', 'You are no longer allowed to register. <br/>If you feel this is a mistake, you may email or call us.');
        return $errors;
    }

    // NON-.COM EMAIL CHECK
    if (!preg_match('/\.com$/i', $email)) {
        if (!$ip_whitelisted) {
            $remaining = arf_increment_failures($ip);
            $errors->add(
                'arf_non_com',
                "Only <b>.com email addresses</b> are allowed. <br/>You have <b><font style='color:#CC0000;'>{$remaining} attempt(s) remaining</font></b> before you are blocked."
            );
        } else {
            $errors->add(
                'arf_non_com',
                "Only <b>.com email addresses</b> are allowed."
            );
        }
        arf_log($ip, $username, $email, $country, 'Blocked: Non .com' . ($ip_whitelisted ? ' (Whitelisted IP)' : ''));
        return $errors;
    }

    // FORBIDDEN KEYWORDS
    $keywords = array_filter(array_map('trim', explode("\n", get_option('arf_banned_keywords'))));
    foreach ($keywords as $kw) {
        if (stripos($username, $kw) !== false || stripos($email, $kw) !== false) {
            $errors->add('arf_keyword_blocked', 'Your username or email contains a forbidden keyword.');
            arf_log($ip, $username, $email, $country, 'Blocked: Forbidden Keyword' . ($ip_whitelisted ? ' (Whitelisted IP)' : ''));
            return $errors;
        }
    }

    // LOG SUCCESS
    arf_log($ip, $username, $email, $country, 'Allowed' . ($ip_whitelisted ? ' (Whitelisted IP)' : ''));

    return $errors;
}

/* ==================================================
 * AUTO-BAN + ALERT
 * ================================================== */
function arf_increment_failures($ip) {
    $threshold = (int) get_option('arf_auto_ban_threshold', 5);
    $fails = (int) get_transient("arf_fail_$ip") + 1;
    set_transient("arf_fail_$ip", $fails, 12 * HOUR_IN_SECONDS);

    if ($fails >= $threshold) {
        arf_add_blocked_ip($ip);
        arf_send_admin_alert($ip);
        return 0;
    }

    return max(0, $threshold - $fails);
}

/* ==================================================
 * DATABASE HELPERS
 * ================================================== */
function arf_is_ip_blocked($ip) {
    global $wpdb;
    return (bool) $wpdb->get_var(
        $wpdb->prepare("SELECT COUNT(*) FROM {$wpdb->prefix}arf_blocked_ips WHERE ip=%s", $ip)
    );
}

function arf_add_blocked_ip($ip) {
    global $wpdb;
    $table = $wpdb->prefix . 'arf_blocked_ips';
    $wpdb->query("CREATE TABLE IF NOT EXISTS $table (
        id BIGINT AUTO_INCREMENT PRIMARY KEY,
        ip VARCHAR(45) UNIQUE
    )");
    $wpdb->insert($table, ['ip' => $ip]);
}

function arf_log($ip, $username, $email, $country, $status) {
    global $wpdb;
    $table = $wpdb->prefix . 'arf_logs';
    $wpdb->query("CREATE TABLE IF NOT EXISTS $table (
        id BIGINT AUTO_INCREMENT PRIMARY KEY,
        ip VARCHAR(45),
        country VARCHAR(10),
        username VARCHAR(60),
        email VARCHAR(100),
        status VARCHAR(255),
        timestamp DATETIME
    )");
    $wpdb->insert($table, [
        'ip' => $ip,
        'country' => $country,
        'username' => $username,
        'email' => $email,
        'status' => $status,
        'timestamp' => current_time('mysql')
    ]);

    // Rotate logs (keep most recent 200)
    $count = (int) $wpdb->get_var("SELECT COUNT(*) FROM $table");
    if ($count > 200) {
        $wpdb->query("DELETE FROM $table ORDER BY timestamp ASC LIMIT " . ($count - 200));
    }
}

/* ==================================================
 * EMAIL ALERT
 * ================================================== */
function arf_send_admin_alert($ip) {
    $emails = array_filter(array_map('trim', explode(',', get_option('arf_admin_alert_emails'))));
    foreach ($emails as $addr) {
        if (is_email($addr)) {
            wp_mail(
                $addr,
                'Registration Firewall Alert',
                "IP auto-banned from registering:\n\nIP: $ip\nDate: " . current_time('mysql'),
                ['Content-Type: text/plain; charset=UTF-8']
            );
        }
    }
}

/* ==================================================
 * UNBLOCK IP (RESET CACHE)
 * ================================================== */
add_action('admin_post_arf_unblock_ip', function () {
    check_admin_referer('arf_unblock_ip');
    $ip = $_GET['ip'];

    global $wpdb;
    $wpdb->delete($wpdb->prefix . 'arf_blocked_ips', ['ip' => $ip]);

    // Reset failure counter
    delete_transient("arf_fail_$ip");

    wp_redirect(admin_url('admin.php?page=arf_admin'));
    exit;
});

/* ==================================================
 * COUNTRY LOOKUP (PLACEHOLDER)
 * ================================================== */
function arf_get_country_from_ip($ip) {
    return 'US'; // Replace with GeoIP if desired
}
?>