<?php

class CloudFw_Gutenberg_Compat {
	/**
	 * Instance.
	 *
	 * @var self
	 */
	public static $instance;

	/**
	 * Backend editor query key.
	 *
	 * @var string
	 */
	public static $backend_query = 'backend-editor';

	/**
	 * Backend editor allowed post types.
	 *
	 * @var array
	 */
	public static $backend_post_types;

	/**
	 * Gutenberg query key.
	 *
	 * @var string
	 */
	public static $gutenberg_query = 'gutenberg-editor';

	/**
	 * Classic editor query key.
	 *
	 * @var string
	 */
	public static $classic_query = 'classic-editor';

	/**
	 * Callback function to check editor.
	 *
	 * @access private
	 * @var string
	 */
	private static $gutenberg_filter_name;

	/**
	 * Callback function to check editor.
	 *
	 * @access private
	 * @var string
	 */
	private static $gutenberg_function_name_for_post_type;

	/**
	 * Constructor.
	 */
	private function __construct() {
		if ( cloudfw_is_core_plugin_installed() && self::get_gutenberg_filter_name() ) {
			add_filter( 'page_row_actions', array( $this, 'add_edit_link' ), 10, 2 );
			add_filter( 'post_row_actions', array( $this, 'add_edit_link' ), 10, 2 );
			add_action( 'admin_print_footer_scripts-edit.php', array( $this, 'edit_dropdown_styles' ), 10 );
			add_action( 'admin_print_footer_scripts-edit.php', array( $this, 'add_new_dropdown' ), 8 );
			add_action( 'admin_print_footer_scripts-post.php', array( $this, 'add_new_dropdown' ), 8 );
			add_action( 'admin_print_footer_scripts-post-new.php', array( $this, 'gutenberg_add_editor_switcher_button' ), 10 );
			add_action( 'admin_print_footer_scripts-post.php', array( $this, 'gutenberg_add_editor_switcher_button' ), 10 );

			if ( cloudfw_is_theme_setting_page( 'editor' ) ) {
				add_filter( 'use_block_editor_for_post', array( $this, 'check_gutenberg' ), 999 );
				add_filter( 'gutenberg_can_edit_post', array( $this, 'check_gutenberg' ), 999 );
			}
		}
	}

	/**
	 * Access the single instance of this class.
	 *
	 * @return self
	 */
	public static function get_instance() {
		if ( ! self::$instance instanceof self ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Gets the gutenberg function.
	 *
	 * @return string
	 */
	public static function get_gutenberg_filter_name() {
		if ( ! isset( self::$gutenberg_filter_name ) ) {
			if ( function_exists( 'use_block_editor_for_post' ) && ! defined( 'GUTENBERG_VERSION' ) ) {
				self::$gutenberg_filter_name = 'use_block_editor_for_post';
			} elseif ( function_exists( 'gutenberg_can_edit_post' ) && defined( 'GUTENBERG_VERSION' ) ) {
				self::$gutenberg_filter_name = 'gutenberg_can_edit_post';
			} else {
				self::$gutenberg_filter_name = false;
			}
		}

		return self::$gutenberg_filter_name;
	}

	/**
	 * Gets the gutenberg function.
	 *
	 * @return string
	 */
	public static function get_gutenberg_function_name_for_post_type() {
		if ( ! isset( self::$gutenberg_function_name_for_post_type ) ) {
			if ( function_exists( 'use_block_editor_for_post_type' ) && ! defined( 'GUTENBERG_VERSION' ) ) {
				self::$gutenberg_function_name_for_post_type = 'use_block_editor_for_post_type';
			} elseif ( function_exists( 'gutenberg_can_edit_post_type' ) && defined( 'GUTENBERG_VERSION' ) ) {
				self::$gutenberg_function_name_for_post_type = 'gutenberg_can_edit_post_type';
			} else {
				self::$gutenberg_function_name_for_post_type = false;
			}
		}

		return self::$gutenberg_function_name_for_post_type;
	}

	/**
	 * Checks if Gutenberg should be disabled.
	 *
	 * @param bool $retval Return val.
	 * @return bool
	 */
	public function check_gutenberg( $retval ) {
		global $post_type;

		$is_backend = isset( $_GET[ self::$backend_query ] ); // WPCS: CSRF ok.

		if ( $is_backend ) {
			return false;
		}

		$editor       = self::get_editor_for_post();
		$is_gutenberg = isset( $_GET[ self::$gutenberg_query ] ); // WPCS: CSRF ok.

		if ( ! $is_gutenberg && 'gutenberg' === $editor ) {
			$is_gutenberg = true;
		}

		if ( $is_gutenberg || ! self::is_backend_editor_for_post_type( $post_type ) ) {
			return $retval;
		}

		return false;
	}

	/**
	 * Gets avaiable editos by post id.
	 *
	 * @param int $post_id The post identifier.
	 * @return array|bool
	 */
	public static function available_editors_for_post( $post_id = null ) {
		$post = get_post( $post_id );

		if ( ! $post instanceof WP_Post ) {
			return false;
		}

		$editor  = self::get_editor_for_post( $post->ID );
		$editors = array(
			'backend'   => self::is_backend_editor_for_post_type( $post->post_type ),
			'gutenberg' => self::is_backend_editor_for_post_type( $post->post_type ) && self::is_gutenberg_editor_for_post_type( $post->post_type ),
		);

		if ( 'gutenberg' !== $editor ) {
			$editors['backend'] = false;
		} else {
			$editors['gutenberg'] = false;
		}

		return $editors;
	}

	/**
	 * Adds specigic editor edit links to the posts row.
	 *
	 * @param array   $actions Post actions.
	 * @param WP_Post $post Edited post.
	 * @return array
	 */
	public function add_edit_link( $actions, $post ) {
		$editor_actions = array();
		$editors        = self::available_editors_for_post( $post->ID );

		if ( empty( $editors ) ) {
			return $actions;
		}

		$title = _draft_or_post_title( $post->ID );

		if ( true === $editors['backend'] ) {
			$backend_url = self::backend_editor_url( $post->ID );

			$editor_actions['backend'] = sprintf(
				'<a href="%s" aria-label="%s">%s</a>',
				esc_url( $backend_url ),
				esc_attr(
					sprintf(
						esc_attr__( 'Edit &#8220;%s&#8221; in the backend editor', 'envision' ),
						$title
					)
				),
				esc_attr__( 'Backend editor', 'envision' )
			);
		}

		if ( true === $editors['gutenberg'] ) {
			$gutenberg_url = self::gutenberg_editor_url( $post->ID );

			$editor_actions['gutenberg'] = sprintf(
				'<a href="%s" aria-label="%s">%s</a>',
				esc_url( $gutenberg_url ),
				esc_attr(
					sprintf(
						esc_attr__( 'Edit &#8220;%s&#8221; in the gutenberg editor', 'envision' ),
						$title
					)
				),
				esc_attr__( 'Gutenberg', 'envision' )
			);
		}

		if ( empty( $editor_actions ) ) {
			return $actions;
		}

		if ( isset( $actions['edit'] ) ) {
			$actions['edit'] = sprintf(
				'<span class="edit-dropdown">%s <span class="edit-dropdown-block">%s</span></span>',
				$actions['edit'],
				implode( '', $editor_actions )
			);
		}

		return $actions;
	}

	/**
	 * Checks whether the backend editor is active for the post type.
	 *
	 * @param string $post_type Post type.
	 * @return boolean
	 */
	public static function is_backend_editor_for_post_type( $post_type ) {
		if ( ! isset( self::$backend_post_types ) ) {
			/**
			 * Backend allowed post types.
			 */
			self::$backend_post_types = apply_filters(
				'cloudfw_composer_default_types', array(
					'page',
				)
			);

			if ( cloudfw_check_onoff( 'blog', 'enable_cc' ) ) {
				self::$backend_post_types[] = 'post';
			}
		}

		return ! empty( $post_type ) && in_array( $post_type, self::$backend_post_types, true );
	}

	/**
	 * Gets backend editor URL.
	 *
	 * @param int $post_id The post identifier.
	 * @return string
	 */
	public static function backend_editor_url( $post_id = null ) {
		return add_query_arg( null, array( self::$backend_query => '' ), get_edit_post_link( $post_id, 'raw' ) );
	}

	/**
	 * Checks if gutenberg editor installed.
	 *
	 * @return bool True if gutenberg editor, False otherwise.
	 */
	public static function is_gutenberg_editor_installed() {
		return false !== self::get_gutenberg_filter_name();
	}

	/**
	 * Gets gutenberg editor URL.
	 *
	 * @param int $post_id The post identifier.
	 * @return string
	 */
	public static function gutenberg_editor_url( $post_id = null ) {
		return add_query_arg( null, array( self::$gutenberg_query => '' ), get_edit_post_link( $post_id, 'raw' ) );
	}

	/**
	 * Checks whether the gutenberg editor is active for the post type.
	 *
	 * @param string $post_type Post type.
	 * @return boolean
	 */
	public static function is_gutenberg_editor_for_post_type( $post_type ) {
		if ( ! self::get_gutenberg_function_name_for_post_type() ) {
			return false;
		}

		return call_user_func( self::$gutenberg_function_name_for_post_type, $post_type );
	}

	/**
	 * Checks whether the gutenberg editor is active for the post.
	 *
	 * @param int $post_id Post id.
	 * @return boolean
	 */
	public static function is_gutenberg_editor_for_post( $post_id = null ) {
		if ( ! self::get_gutenberg_filter_name() ) {
			return false;
		}

		if ( empty( $post_id ) ) {
			global $post;

			if ( $post ) {
				$post_id = $post->ID;
			} else {
				return false;
			}
		}

		return call_user_func( self::$gutenberg_filter_name, $post_id );
	}

	/**
	 * Gets editor type.
	 *
	 * @param int $post_id The post identifier.
	 * @return string Possible return values: "classic", "backend".
	 */
	public static function get_editor_for_post( $post_id = null ) {
		if ( empty( $post_id ) ) {
			$post_id = get_the_ID();
		}

		$is_editor            = _check_onoff( cloudfw_composer_is_activated( $post_id ) );
		$autodetect_gutenberg = true;

		if ( isset( $_GET[ self::$backend_query ] ) ) { // WPCS: CSRF ok.
			$editor = 'backend';
		} elseif ( isset( $_GET[ self::$classic_query ] ) ) { // WPCS: CSRF ok.
			$editor = 'classic';
			$autodetect_gutenberg = false;
		} elseif ( isset( $_GET[ self::$gutenberg_query ] ) ) { // WPCS: CSRF ok.
			$editor = 'gutenberg';
		} elseif ( ! $is_editor ) {
			$editor = 'classic';
		} elseif ( $is_editor ) {
			$editor = 'backend';
		} else {
			$editor = in_array( get_post_type( $post_id ), apply_filters( 'cloudfw_editor_get_type', array() ), true ) ? 'backend' : 'classic';
		}

		if ( self::is_gutenberg_editor_installed() && $autodetect_gutenberg && ( empty( $editor ) || 'classic' === $editor ) && self::is_gutenberg_editor_for_post_type( get_post_type( $post_id ) ) ) {
			$editor = 'gutenberg';
		}

		return apply_filters( 'cloudfw_editor_get_type', $editor, $post_id );
	}

	/**
	 * Adds dropdown styles.
	 */
	public static function edit_dropdown_styles() {
		?>
		<style type="text/css">
			.edit-dropdown {
				position: relative;
			}

			.edit-dropdown > a {
				z-index: 3;
				position: relative;
			}

			.edit-dropdown:hover > a {
			}

			.edit-dropdown:before {
				display: none;
				content: "";
				position: absolute;
				top: -4px;
				right: -5px;
				bottom: -4px;
				left: -10px;
				z-index: 2;
				background: #f7f7f7;
				border: 1px solid #ccc;
				border-bottom: 0;
				border-top-left-radius: 2px;
				border-top-right-radius: 2px;
			}

			.edit-dropdown:hover:before {
				display: block;
			}

			.edit-dropdown-block {
				display: none;
				position: absolute;
				left: -10px;
				top: calc(100% + 3px);
				background: #f7f7f7;
				border: 1px solid #ccc;
				box-shadow: 0 3px 20px rgba(0,0,0,.2);
				z-index: 1;
				padding: 5px 0;
				border-radius: 2px;
				border-top-left-radius: 0;
			}

			.edit-dropdown-block > a {
				display: block;
				white-space: nowrap;
				padding: 5px 10px;
				border-bottom: 1px solid #ccc;
			}

			.edit-dropdown-block > a:hover {
				border-bottom-color: transparent;
				background: #00a0d2;
				color: #fff;
			}

			.edit-dropdown-block > a:last-child {
				border-bottom: 0;
			}

			.edit-dropdown:hover .edit-dropdown-block {
				display: block;
			}
		</style>
		<?php
	}

	/**
	 * Adds dropdown styles.
	 */
	public static function add_new_dropdown() {
		global $typenow;

		if ( ! self::is_gutenberg_editor_for_post_type( $typenow ) ) {
			return;
		}

		$editors = array(
			'backend'   => self::is_backend_editor_for_post_type( $typenow ),
			'gutenberg' => self::is_backend_editor_for_post_type( $typenow ) && self::is_gutenberg_editor_for_post_type( $typenow ),
		);

		?>
		<script type="text/javascript">
			document.addEventListener( 'DOMContentLoaded', function() {
				var buttons = document.getElementsByClassName( 'page-title-action' ),
					button = buttons.item( 0 );

				if ( ! button ) {
					return;
				}

				var url = button.href;
				var urlHasParams = ( -1 !== url.indexOf( '?' ) );
				var newbutton = '';
				var addButton = false;

				newbutton += '<span id="split-page-title-action" class="split-page-title-action">';
				newbutton += '<a href="' + url + '">' + button.innerText + '</a>';
				newbutton += '<span class="expander" tabindex="0" role="button" aria-haspopup="true" aria-label="<?php echo esc_js( __( 'Toggle editor selection menu', 'envision' ) ); ?>"></span>';

				newbutton += '<span class="dropdown">';
					<?php if ( true === $editors['backend'] ) : ?>
					addButton = true;
					var backendUrl = url + ( urlHasParams ? '&' : '?' ) + "<?php echo esc_js( self::$backend_query ); ?>";
					newbutton += '<a href="' + backendUrl + '"><?php echo esc_js( __( "Content Composer", 'envision' ) ); ?></a>';
					<?php endif; ?>

					<?php if ( true === $editors['gutenberg'] ) : ?>
					addButton = true;
					var gutenbergUrl = url + ( urlHasParams ? '&' : '?' ) + "<?php echo esc_js( self::$gutenberg_query ); ?>";
					newbutton += '<a href="' + gutenbergUrl + '">Gutenberg</a>';
					<?php endif; ?>

				newbutton += '</span></span><span class="page-title-action" style="display:none;"></span>';

				if ( ! addButton ) {
					return false;
				}

				button.insertAdjacentHTML( 'afterend', newbutton );
				button.parentNode.removeChild( button );

				var expander = document.getElementById( 'split-page-title-action' ).getElementsByClassName( 'expander' ).item( 0 );
				var dropdown = expander.parentNode.querySelector( '.dropdown' );
				function toggleDropdown() {
					dropdown.classList.toggle( 'visible' );
				}
				expander.addEventListener( 'click', function( e ) {
					e.preventDefault();
					toggleDropdown();
				} );
				expander.addEventListener( 'keydown', function( e ) {
					if ( 13 === e.which || 32 === e.which ) {
						e.preventDefault();
						toggleDropdown();
					}
				} );
			} );
		</script>
		<style type="text/css">
			.split-page-title-action {
				display: inline-flex;
				position: relative;
				top: -3px;
				align-items: center;
			}

			.split-page-title-action a,
			.split-page-title-action a:active,
			.split-page-title-action .expander:after {
				padding: 6px 10px;
				position: relative;
				text-decoration: none;
				border: 1px solid #ccc;
				border-radius: 2px;
				background: #f7f7f7;
				text-shadow: none;
				font-weight: 600;
				font-size: 13px;
				line-height: normal; /* IE8-IE11 need this for buttons */
				color: #0073aa; /* some of these controls are button elements and don't inherit from links */
				cursor: pointer;
				outline: 0;
			}

			.split-page-title-action a:hover,
			.split-page-title-action .expander:hover:after {
				border-color: #008EC2;
				background: #00a0d2;
				color: #fff;
			}

			.split-page-title-action a:focus,
			.split-page-title-action .expander:focus:after {
				border-color: #5b9dd9;
				box-shadow: 0 0 2px rgba( 30, 140, 190, 0.8 );
			}

			.split-page-title-action .expander:after {
				content: "\f140";
				font: 400 20px/.5 dashicons;
				speak: none;
				top: 0;
				<?php if ( is_rtl() ) : ?>
				right: -1px;
				<?php else : ?>
				left: -1px;
				<?php endif; ?>
				position: relative;
				vertical-align: top;
				text-decoration: none !important;
				padding: 4px 5px 4px 3px;
				line-height: 1.5;
			}

			.split-page-title-action .dropdown {
				display: none;
			}

			.split-page-title-action .dropdown.visible {
				display: block;
				position: absolute;
				z-index: 1;
				top: 100%;
    			margin-top: -1px;
			}

			.split-page-title-action .dropdown.visible a {
				display: block;
				top: 0;
				margin: -1px 0;
				<?php if ( is_rtl() ) : ?>
				padding-left: 9px;
				<?php else : ?>
				padding-right: 9px;
				<?php endif; ?>
			}

			.split-page-title-action .expander {
				outline: none;
			}

		</style>
		<?php
	}

	/**
	 * Adds editor switcher button.
	 */
	public function gutenberg_add_editor_switcher_button() {
		global $post, $post_type;

		if ( ! is_object( $post ) ) {
			return;
		}

		if ( self::is_backend_editor_for_post_type( $post_type ) ) {
			if ( 'gutenberg' === self::get_editor_for_post( $post->ID ) ) {
				$link        = self::backend_editor_url();
				$label       = esc_attr__( 'Switch to Content Composer', 'envision' );
				$button_html = '<a href="' . $link . '" class="button button-primary button-large components-button">' . $label . '</a>'; // WPCS: XSS ok.
				?>
				<script type="text/javascript">
					jQuery( function() {
						var isButtonAdded = false;

						window.wp.data.subscribe( function() {
							if ( false !== isButtonAdded ) {
								return;
							}

							setTimeout( function() {
								var toolbar = jQuery( '.edit-post-header__settings' );

								if ( toolbar.length ) {
									toolbar.prepend( '<?php echo wp_kses_post($button_html); // WPCS: XSS ok. ?>' );
								}
							}, 1 );

							isButtonAdded = true;
						} );
					} );

				</script>
				<?php
			}
		}
	}
}
