<?php
/**
 * Envision Core Plugin class.
 */
class CloudFw_Envision_Core {

    /**
     * Construct.
     */
    public function __construct() {
        register_activation_hook(__FILE__, array( $this, 'activate' ));
        register_deactivation_hook(__FILE__, array( $this, 'deactivate' ));

        add_action( 'network_admin_notices', array( $this, 'admin_notices' ) );
        add_action( 'admin_notices', array( $this, 'admin_notices' ) );
        add_action( 'plugins_loaded', array( $this, 'load_textdomain' ) );
        add_action( 'init', array( $this, 'register_portfolio' ) );
        add_filter( 'the_content', array( $this, 'pre_process_shortcode' ), 7);
        add_filter( 'cloudfw_load_shortcodes_class', array( $this, 'load_shortcodes_class' ));
        add_filter( 'cloudfw_register_metabox', array( $this, 'load_metabox_class' ));
        add_action( 'cloudfw_register_widgets', array( $this, 'register_widgets' ) );
        add_action( 'cloudfw_register_shortcodes', array( $this, 'register_shortcodes' ) );
    }

    /**
     * Checks if the theme installed.
     *
     * @return boolean
     */
    public function is_theme_installed() {
        return defined('CLOUDFW_TMP_PATH') && ! empty( CLOUDFW_TMP_PATH );
    }

    /**
     * Admin notice.
     *
     * @return void
     */
    public function admin_notices() {
        if ( $this->is_theme_installed() ) {
            return;
        }
        ?>
        <div class="notice notice-error">
            <p><?php esc_html_e( 'Envision Core Plugin only works with Envision theme. Please install Envision theme first.', 'envision-core' ); ?></p>
        </div>
        <?php
    }

    /**
     * Loads the textdomain file if exists in the languages folder.
     */
    public function load_textdomain(){
        load_plugin_textdomain( 'envision-core', false, path_join( plugin_basename( CLOUDFW_ENVISION_CORE_PATH ), 'languages/' ) );
    }

    /**
     * Loads shortcodes main class.
     */
    public function load_shortcodes_class(){
        require_once plugin_dir_path(__FILE__) . 'includes/classes/class-cloudfw-shortcodes.php';
        $GLOBALS['CloudFw_Shortcodes'] = new CloudFw_Shortcodes();
    }

    /**
     * Loads metabox class.
     */
    public function load_metabox_class(){
        require_once plugin_dir_path(__FILE__) . 'includes/classes/class-cloudfw-metabox.php';
        $GLOBALS['CloudFw_Shortcodes'] = new CloudFw_Shortcodes();
    }

    /**
     * Register theme widgets.
     *
     * @return void
     */
    public function register_shortcodes() {
        cloudfw_autoload_folders( path_join( plugin_dir_path(__FILE__), 'includes/shortcodes' ) );
    }

    /**
     * Registers theme widgets.
     *
     * @return void
     */
    public function register_widgets() {
        cloudfw_autoload_folders( path_join( plugin_dir_path(__FILE__), 'includes/widgets' ) );
    }

    /**
     * Registers portfolio post type and taxonomies.
     */
    public function register_portfolio() {
        if ( function_exists("cloudfw_get_option") ) {
            $post_slug = sanitize_title( cloudfw_get_option( 'portfolio',  'slug', 'portfolio' ) );
            $category_slug = sanitize_title( cloudfw_get_option( 'portfolio',  'category_slug', 'portfolio-category' ) );
            $filter_slug = sanitize_title( cloudfw_get_option( 'portfolio',  'filter_slug', 'portfolio-filter' ) );
            $tag_slug = sanitize_title( cloudfw_get_option( 'portfolio',  'tag_slug', 'portfolio-tags' ) );
        } else {
            $post_slug = sanitize_title( 'portfolio' );
            $category_slug = sanitize_title( 'portfolio-category' );
            $filter_slug = sanitize_title( 'portfolio-filter' );
            $tag_slug = sanitize_title( 'portfolio-tags' );
        }

        /** Register Custom Post Type */
        register_post_type('portfolio', array(
            'label'         => __('Portfolio Post', 'envision-core'),
            'public'        => true,
            'show_ui'       => true,
            'query_var'     => 'portfolio',
            'supports'      => array(
                'title',
                'editor',
                'thumbnail',
                'comments',
            ),
            'rewrite'       => array(
                'slug'          => $post_slug,
                'with_front'    =>  true
            ),
            'menu_icon'     => 'dashicons-schedule',
            'menu_position' => apply_filters( 'cloudfw_portfolio_menu_position', 29 ),
            'hierarchical'  => false,
            'labels'        => array(
                'name'               => __('Portfolio', 'envision-core'),
                'singular_name'      => __('Portfolio', 'envision-core'),
                'add_new'            => __('Add New', 'envision-core'),
                'add_new_item'       => __('Add New Portfolio Post', 'envision-core'),
                'edit'               => __('Edit Portolio Post', 'envision-core'),
                'edit_item'          => __('Edit Portolio Post', 'envision-core'),
                'new_item'           => __('New Portolio Post', 'envision-core'),
                'view'               => __('View Post', 'envision-core'),
                'view_item'          => __('View Post', 'envision-core'),
                'not_found'          => __('There is no any portolio post', 'envision-core'),
                'not_found_in_trash' => __('There is no any portolio post in the trash', 'envision-core')
            )
        ));

        /** Category Labels */
        $portfolio_category_labels = array(
            'singular_name'          => __('Category', 'envision-core'),
            'name'                   => __('Portfolio Category', 'envision-core'),
            'search_items'           => __('Search Category', 'envision-core'),
            'all_items'              => __('All Categories', 'envision-core'),
            'parent_item'            => __('Parent Category', 'envision-core'),
            'parent_item_colon'      => __('Parent Category', 'envision-core'),
            'edit_item'              => __('Edit Category', 'envision-core'),
            'update_item'            => __('Update Category', 'envision-core'),
            'add_new_item'           => __('Add New Category', 'envision-core'),
            'new_item_name'          => __('New Category Name', 'envision-core'),
            'menu_name'              => __('Categories', 'envision-core')
        );

        /** Register Category */
        register_taxonomy(
            'portfolio-category',
            array('portfolio'),
            array(
                'hierarchical' => true,
                'labels'       => $portfolio_category_labels,
                'show_ui'      => true,
                'query_var'    => true,
                'rewrite'      => array(
                    'slug' => apply_filters( 'cloudfw_portfolio_category_slug', $category_slug ),
                )
        ));

        /** Category Labels */
        $portfolio_filter_labels = array(
            'singular_name'          => __('Filter', 'envision-core'),
            'name'                   => __('Portfolio Filter', 'envision-core'),
            'search_items'           => __('Search Filter', 'envision-core'),
            'all_items'              => __('All Filters', 'envision-core'),
            'parent_item'            => __('Parent Filter', 'envision-core'),
            'parent_item_colon'      => __('Parent Filter', 'envision-core'),
            'edit_item'              => __('Edit Filter', 'envision-core'),
            'update_item'            => __('Update Filter', 'envision-core'),
            'add_new_item'           => __('Add New Filter', 'envision-core'),
            'new_item_name'          => __('New Filter Name', 'envision-core'),
            'menu_name'              => __('Filters', 'envision-core')
        );
        register_taxonomy(
            'portfolio-filter',
            array('portfolio'),
            array(
                'hierarchical' => true,
                'labels'       => $portfolio_filter_labels,
                'show_ui'      => true,
                'query_var'    => true,
                'rewrite'      => array(
                    'slug' => apply_filters( 'cloudfw_portfolio_filter_slug', $filter_slug ),
                )
        ));

        /** Category Labels */
        $portfolio_tag_labels = array(
            'singular_name'          => __('Related Tag', 'envision-core'),
            'name'                   => __('Portfolio Related Tag', 'envision-core'),
            'search_items'           => __('Search Related Tag', 'envision-core'),
            'all_items'              => __('All Related Tags', 'envision-core'),
            'parent_item'            => __('Parent Related Tag', 'envision-core'),
            'parent_item_colon'      => __('Parent Related Tag', 'envision-core'),
            'edit_item'              => __('Edit Related Tag', 'envision-core'),
            'update_item'            => __('Update Related Tag', 'envision-core'),
            'add_new_item'           => __('Add New Related Tag', 'envision-core'),
            'new_item_name'          => __('New Related Tag Name', 'envision-core'),
            'menu_name'              => __('Related Tags', 'envision-core')
        );
        register_taxonomy(
            'portfolio-tags',
            array('portfolio'),
            array(
                'hierarchical' => false,
                'labels'       => $portfolio_tag_labels,
                'show_ui'      => true,
                'query_var'    => true,
                'rewrite'      => array(
                    'slug' => apply_filters( 'cloudfw_portfolio_tag_slug', $tag_slug ),
                )
        ));
    }

    /**
     * Content filter.
     *
     * @param string $content
     * @return string
     */
    public function pre_process_shortcode( $content ) {
        if ( ! $this->is_theme_installed() ) {
            return $content;
        }

        global $cloudfw_pre_do_shortcodes, $shortcode_tags, $orig_shortcode_tags;

        // Backup current registered shortcodes and clear them all out
        $orig_shortcode_tags = $shortcode_tags;
        remove_all_shortcodes();

        foreach ((array)$orig_shortcode_tags as $shortcode => $callback) {
            if ( in_array($shortcode, $cloudfw_pre_do_shortcodes) ) {
                add_shortcode( $shortcode, $callback );
            }
        }

        // Do the shortcode (only the selected shortcodes)
        $content = do_shortcode($content);

        // Put the original shortcodes back
        $shortcode_tags = $orig_shortcode_tags;

        return $content;
    }

    /**
     * Plugin activation actions.
     *
     * @return void
     */
    public function activate() {}

    /**
     * Plugin deactivation actions.
     *
     * @return void
     */
    public function deactivate(){}
}